/**
 * \file caam_op_rng.c
 *
 * \brief Architecture specific implementation of rng functions
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2016 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

#include <stdlib.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <string.h>
#include "caam.h"
#include <linux/caam_ee.h>

static sdc_error_t caam_rng_reseed_gen (sdc_session_t *session,
                                        const bool reseed, const uint8_t *seed_data, const size_t seed_len,
                                        const size_t gen_len, uint8_t *gen_data)
{
    sdc_error_t err = SDC_OK;
    uint8_t *gen_data_ptr;
    size_t remaining_gen_len;
    struct caam_ee_rng_params iodata;

    uint8_t seed_buf[CAAM_EE_RNG_RESEED_LEN];

    int res;

    gen_data_ptr = gen_data;
    remaining_gen_len = gen_len;

    memset(&iodata, 0, sizeof(iodata));

    if (reseed) {
        iodata.rng_flags |= CAAM_EE_RNG_FLAG_RESEED;

        if (seed_len > 0) {
            iodata.reseed_buf = &seed_buf[0];

            if (seed_len >= CAAM_EE_RNG_RESEED_LEN) {
                memcpy(seed_buf, seed_data, CAAM_EE_RNG_RESEED_LEN);
            } else {
                memset(seed_buf, 0, CAAM_EE_RNG_RESEED_LEN);
                memcpy(seed_buf, seed_data, seed_len);
            }
        }
    }

    if (remaining_gen_len > 0) {
        iodata.rng_flags |= CAAM_EE_RNG_FLAG_GETRNG;
    }

    while ((iodata.rng_flags != 0) && (err == SDC_OK)) {
        iodata.random_out = gen_data_ptr;
        iodata.random_out_len = remaining_gen_len;
        if (remaining_gen_len > CAAM_EE_MAX_RANDOM_LEN) {
            iodata.random_out_len = CAAM_EE_MAX_RANDOM_LEN;
        }

        res = ioctl(session->arch.fd, CAAM_RNG, &iodata);

        if (res == 0) {
            remaining_gen_len -= iodata.random_out_len;
            gen_data_ptr += iodata.random_out_len;
        } else {
            err = error_from_ioctl_errno(errno, CAAM_RNG, iodata.rng_flags);
        }

        iodata.reseed_buf = NULL;
        iodata.rng_flags = 0;
        if (remaining_gen_len > 0) {
            iodata.rng_flags = CAAM_EE_RNG_FLAG_GETRNG;
        }
    }

    return err;

}

sdc_error_t sdc_arch_random_gen (sdc_session_t *session, const size_t len, uint8_t *random_data)
{
    return caam_rng_reseed_gen(session, false, NULL, 0, len, random_data);
}

